# 🎮 ULTIMATE MULTIPLAYER WEB SHOOTER - SETUP GUIDE

## 📦 Complete Project Files Included

This is a production-ready multiplayer FPS/TPS web game with all source code.

### Files Included:
1. **server.js** - Node.js WebSocket server (multiplayer backend)
2. **game.html** - HTML5 client game application
3. **package.json** - Dependencies and scripts
4. **weapons.json** - Weapon configurations and balancing
5. **maps.json** - Map designs and level layouts
6. **README.md** - Full documentation

---

## 🚀 INSTALLATION & SETUP

### Step 1: Install Node.js
- Download from https://nodejs.org/ (v14 or higher)
- Verify installation: `node --version`

### Step 2: Setup Project

```bash
# Navigate to project directory
cd ultimate-multiplayer-shooter

# Install all dependencies
npm install
```

### Step 3: Start Server

```bash
# Start the server
npm start

# Expected output:
# ╔════════════════════════════════════════════════════════╗
# ║   MULTIPLAYER WEB SHOOTER - SERVER                    ║
# ║   Server running on: ws://localhost:8080              ║
# ║   API running on: http://localhost:8080               ║
# ╚════════════════════════════════════════════════════════╝
```

### Step 4: Play the Game

Open in your browser:
```
http://localhost:8080/game.html
```

---

## 🎮 GAMEPLAY GUIDE

### Controls
- **WASD** - Move
- **SHIFT** - Sprint
- **SPACE** - Jump
- **CTRL** - Crouch
- **Mouse** - Aim
- **Click** - Fire
- **R** - Reload
- **1-5** - Switch weapons

### Game Features
✅ Real-time multiplayer (up to 8 players per room)
✅ 7 different weapons with unique properties
✅ 3 detailed map designs
✅ Server-authoritative anti-cheat system
✅ Live scoreboard and kill feed
✅ Realistic damage model with armor
✅ Weapon recoil and bullet spread
✅ Headshot multipliers (2x damage)

---

## 🔧 ARCHITECTURE OVERVIEW

### Server Architecture
```
WebSocket Server (Port 8080)
├── Room Management
│   ├── Create/Join/Leave Rooms
│   ├── Player Synchronization
│   └── Match Data Tracking
├── Game Logic
│   ├── Input Validation
│   ├── Damage Calculation
│   ├── Anti-Cheat Detection
│   └── Physics Updates
└── REST API
    ├── Authentication (/api/auth/login)
    ├── Room Management (/api/rooms/*)
    └── Configuration (/api/config/*)
```

### Client Architecture
```
Game Engine
├── Input System
│   └── Keyboard, Mouse, Movement
├── Rendering
│   ├── 2.5D Perspective Drawing
│   ├── Player Rendering
│   ├── Environment
│   └── Particle Effects
├── Game State
│   ├── Player Data
│   ├── Weapon System
│   └── Match Data
├── Network
│   └── WebSocket Communication
└── UI
    ├── HUD (Health, Ammo)
    ├── Scoreboard
    ├── Kill Feed
    └── Radar
```

---

## 📊 GAME BALANCE

### Weapon Stats

| Weapon | Damage | Fire Rate | Range | Accuracy | Usage |
|--------|--------|-----------|-------|----------|-------|
| M4A1 | 28 | 750 RPM | 800m | High | Best all-rounder |
| AK-47 | 34 | 600 RPM | 750m | Medium | High damage |
| MP5 | 18 | 950 RPM | 350m | Medium | Close quarters |
| Shotgun | 75 | 80 RPM | 200m | Low | One-shot potential |
| Sniper | 90 | 40 RPM | 1500m | Very High | Long range |
| Pistol | 22 | 400 RPM | 500m | Medium | Backup weapon |

### Damage Multipliers
- **Headshot**: 2.0x
- **Body**: 1.0x
- **Legs**: 0.75x
- **Armor**: Reduces damage by 25-75%

---

## 🌐 NETWORKING DETAILS

### Server Tick Rate
- **60 TPS** - Server game loop (entity updates, physics, input processing)
- **30 TPS** - Player state broadcasts to clients (bandwidth optimization)
- **Anti-Cheat** - Real-time validation of player positions and shots

### Network Packet Types

**Client to Server:**
- Player Input (60 Hz) - Movement, aim, shooting
- Weapon Fire (on demand) - Validated server-side
- Chat Messages

**Server to Client:**
- Game State Updates (30 Hz) - All player positions
- Hit Notifications - Damage events
- Health/Ammo Updates
- Scoreboard Changes

### Latency Tolerance
- **Optimal**: <100ms
- **Acceptable**: <200ms
- **Server**: Compensates for latency with prediction

---

## 🔒 SECURITY & ANTI-CHEAT

### Server-Authoritative Systems
✅ **Hit Detection** - Only server validates shots
✅ **Damage Calculation** - Server calculates all damage
✅ **Position Validation** - Checks for speed hacks
✅ **Ammo Management** - Server controls ammunition
✅ **Weapon Range** - Enforces max weapon range

### Anti-Cheat Detection
- Speed hack detection (comparing movement distances)
- Invalid position detection (checking boundary violations)
- Suspicious action tracking (flagging unusual patterns)
- JWT token validation (prevents session hijacking)

---

## 📈 PERFORMANCE OPTIMIZATION

### Client-Side
- Object pooling for bullets and particles
- Canvas rendering optimization
- Efficient 2.5D perspective calculation
- Dynamic update rates based on network conditions

### Server-Side
- Room-based state management
- Efficient data serialization
- Tick-based physics updates
- Memory pooling for game objects

### Target Performance
- **FPS**: 60+ on mid-range devices
- **Memory**: ~100MB server, ~50MB client
- **Network**: ~10KB/s per player
- **CPU**: <20% at 8 players

---

## 🎨 CUSTOMIZATION

### Adding a New Weapon
1. **Edit weapons.json**:
```json
{
  "your_weapon": {
    "name": "Weapon Name",
    "damage": 30,
    "fireRate": 750,
    "range": 500,
    "magazine": 30,
    "reloadTime": 2.0,
    "description": "Description"
  }
}
```

2. **Add to server.js** weapon registry
3. **Update client UI** with new weapon selector

### Adding a New Map
1. **Edit maps.json** and add map entry:
```json
{
  "id": "new_map",
  "name": "Map Name",
  "spawnPoints": [...],
  "staticObjects": [...],
  "weaponPickups": [...]
}
```

2. **Test in-game** for balance and flow

---

## 🚀 DEPLOYMENT OPTIONS

### Option 1: Heroku (Easiest)
```bash
# Login to Heroku
heroku login

# Create app
heroku create your-game-name

# Deploy
git push heroku main

# View live
heroku open
```

### Option 2: VPS (DigitalOcean, AWS, Linode)
```bash
# SSH into VPS
ssh root@your_vps_ip

# Install Node.js
curl -fsSL https://deb.nodesource.com/setup_18.x | sudo -E bash -
sudo apt install -y nodejs

# Clone repo
git clone <repo-url>
cd ultimate-multiplayer-shooter
npm install

# Start with PM2 (keeps running)
npm install -g pm2
pm2 start server.js --name "shooter"
pm2 startup
pm2 save
```

### Option 3: Docker
```bash
# Build image
docker build -t multiplayer-shooter .

# Run container
docker run -p 8080:8080 multiplayer-shooter

# Docker Compose
docker-compose up
```

### Option 4: Railway/Render
- Connect GitHub repository
- Auto-deploy on push
- Free SSL certificates

---

## 🧪 TESTING

### Load Testing
```bash
# Test with 100 concurrent players
# Use tool like Artillery or Apache JMeter
artillery quick --count 100 --duration 60 http://localhost:8080
```

### Manual Testing
1. Open multiple browser windows/tabs
2. Create different usernames
3. Join same room
4. Test gameplay mechanics
5. Check console for errors

---

## 📱 MOBILE SUPPORT

Current version optimized for desktop. For mobile:

1. Implement touch controls
2. Add on-screen joystick
3. Reduce canvas resolution
4. Optimize network packets
5. Test on mobile browsers

---

## 🐛 TROUBLESHOOTING

### Server Won't Start
```bash
# Check if port 8080 is in use
lsof -i :8080
# Kill process if needed
kill -9 <PID>
```

### WebSocket Connection Fails
- Ensure server is running
- Check firewall settings
- Verify client connects to correct IP
- Check browser console for errors

### Game Lags/Stutters
- Reduce number of players
- Check network latency
- Close other applications
- Update graphics drivers

### Players Not Synchronized
- Check server logs for errors
- Verify WebSocket connection
- Restart server and client
- Clear browser cache

---

## 📚 API DOCUMENTATION

### REST Endpoints

**POST /api/auth/login**
```javascript
Request:
{
  "username": "PlayerName"
}

Response:
{
  "playerId": "player_12345",
  "username": "PlayerName",
  "token": "eyJhbGc..."
}
```

**POST /api/rooms/create**
```javascript
Request:
{
  "isPrivate": false
}

Response:
{
  "roomId": "ABC123",
  "isPrivate": false,
  "playersCount": 0,
  "maxPlayers": 8
}
```

**GET /api/rooms/list**
```javascript
Response:
[
  {
    "roomId": "ABC123",
    "playersCount": 3,
    "maxPlayers": 8,
    "mapId": "arena_01"
  }
]
```

---

## 📞 SUPPORT & CONTACT

For issues, questions, or contributions:
- Create issue on GitHub
- Check documentation in README.md
- Review server logs for errors
- Test in different browsers

---

## 📄 LICENSE

MIT License - Free for personal and commercial use

---

**Ready to play? Start the server and jump into battle! 🎮**

```bash
npm start
# Then open http://localhost:8080/game.html
```
