# 🏗️ TACTICAL WARFARE - Architecture & Design Document

## Table of Contents
1. [System Architecture](#system-architecture)
2. [Network Architecture](#network-architecture)
3. [Game Loop](#game-loop)
4. [Data Structures](#data-structures)
5. [State Management](#state-management)
6. [Anti-Cheat](#anti-cheat)
7. [Performance Optimization](#performance-optimization)
8. [Scalability](#scalability)

---

## System Architecture

### High-Level Overview

```
┌──────────────────────────────────────────────────────────────┐
│                     CLIENT LAYER                             │
│  ┌────────────────────────────────────────────────────────┐ │
│  │  Web Browser (HTML5 Canvas)                            │ │
│  │  ┌──────────────────────────────────────────────────┐ │ │
│  │  │ GameClient Class                                │ │ │
│  │  │ - Input Handler                                 │ │ │
│  │  │ - Renderer                                      │ │ │
│  │  │ - Network Manager                              │ │ │
│  │  │ - Client Prediction                            │ │ │
│  │  └──────────────────────────────────────────────────┘ │ │
│  └────────────────────────────────────────────────────────┘ │
└───────────────────────┬──────────────────────────────────────┘
                        │ WebSocket (Socket.io)
                        │ Binary/JSON Messages
                        │
┌───────────────────────┴──────────────────────────────────────┐
│                     NETWORK LAYER                            │
│  ┌────────────────────────────────────────────────────────┐ │
│  │ Express.js HTTP Server + Socket.io                     │ │
│  │ - Message Routing                                      │ │
│  │ - Room Management                                      │ │
│  │ - Broadcast System                                     │ │
│  └────────────────────────────────────────────────────────┘ │
└───────────────────────┬──────────────────────────────────────┘
                        │
┌───────────────────────┴──────────────────────────────────────┐
│                    SERVER LAYER                              │
│  ┌────────────────────────────────────────────────────────┐ │
│  │ Game Logic & State Management                          │ │
│  │ ┌──────────────────────────────────────────────────┐  │ │
│  │ │ Game Loop (60 Hz Tick)                          │  │ │
│  │ │ - Physics Update                                │  │ │
│  │ │ - Collision Detection                           │  │ │
│  │ │ - Hit Validation (Server Auth.)                 │  │ │
│  │ │ - State Synchronization                         │  │ │
│  │ └──────────────────────────────────────────────────┘  │ │
│  │ ┌──────────────────────────────────────────────────┐  │ │
│  │ │ Room Manager                                     │  │ │
│  │ │ - Player Sessions                               │  │ │
│  │ │ - Matchmaking                                   │  │ │
│  │ │ - Game Rules                                    │  │ │
│  │ └──────────────────────────────────────────────────┘  │ │
│  │ ┌──────────────────────────────────────────────────┐  │ │
│  │ │ Anti-Cheat Module                               │  │ │
│  │ │ - Hit Detection                                 │  │ │
│  │ │ - Position Validation                           │  │ │
│  │ │ - Behavior Analysis                             │  │ │
│  │ └──────────────────────────────────────────────────┘  │ │
│  └────────────────────────────────────────────────────────┘ │
└──────────────────────────────────────────────────────────────┘
```

---

## Network Architecture

### Socket.io Event Flow

#### Join Game Flow
```
CLIENT                              SERVER
  │
  ├─ joinGame()
  │  {username, roomCode}
  │                              ┌─ Validate username
  │                              ├─ Find or create room
  │                              ├─ Create Player instance
  │                              ├─ Add to room
  │                              └─ Start game if enough players
  │
  ├─ onPlayerJoined ◄─────────────────┤
  │  {player, roomState}              │
  │                                   │
  ├─ Broadcast to other players ◄─────┤
  │                                   │
  └─ Begin rendering game            │
```

#### Input → State → Broadcast Flow
```
CLIENT (60 FPS)         SERVER (60 Hz Tick)        OTHER CLIENTS
    │                           │                        │
    ├─ Read inputs              │                        │
    │  (WASD, mouse)            │                        │
    │                           │                        │
    ├─ Client-side              │                        │
    │  prediction                │                        │
    │                           │                        │
    ├─ playerInput event        │                        │
    │  ──────────────────────► │                        │
    │                           ├─ Update player state   │
    │                           ├─ Validate position     │
    │                           ├─ Check bounds          │
    │                           │                        │
    │                           ├─ gameState event       │
    │                           ├─────────────────────►  │
    │                           │                        ├─ Interpolate
    │                           │                        │  positions
    │                           │                        │
    │                           │                        ├─ Render frame
```

#### Shooting Flow
```
CLIENT                      SERVER                    TARGET CLIENT
  │
  ├─ Click to fire          
  │                         
  ├─ Local muzzle flash   
  │  (prediction)          
  │                        
  ├─ shoot event           
  │  {weaponId, angle}      
  │─────────────────────► ├─ Validate weapon
  │                        ├─ Check ammo
  │                        ├─ Hitscan detection
  │                        ├─ Calculate damage
  │                        ├─ Server-side hit check
  │                        ├─ Broadcast hit
  │                        │
  │                        ├─ bulletFired event
  │                        ├─────────────────►  ├─ Play impact FX
  │                        │                    ├─ Play sound
  │                        │                    └─ Update health bar
  │                        │
  │                        └─ gameState update
  │                           (new health, kills)
```

---

## Game Loop

### Server Tick (60 Hz)

```javascript
setInterval(() => {
  // 1. RECEIVE PHASE
  // Process all pending socket events from clients
  //   - playerInput events
  //   - shoot events
  //   - reload events
  
  // 2. UPDATE PHASE
  rooms.forEach(room => {
    // Update player positions & states
    room.players.forEach(player => {
      player.update(inputBuffer);
      validatePosition(player);  // Anti-cheat
    });
    
    // Collision detection
    checkCollisions(room.players);
    
    // Physics simulation
    applyGravity();
    applyFriction();
  });
  
  // 3. BROADCAST PHASE
  rooms.forEach(room => {
    // Send game state to all clients in room
    io.to(room.id).emit('gameState', {
      players: room.players.map(p => p.getState()),
      timestamp: Date.now(),
      roundTime: room.getRoundTime()
    });
  });
  
}, 1000 / 60); // 16.67ms per tick
```

### Client Frame (60 FPS)

```javascript
function gameLoop() {
  // 1. INPUT PHASE
  const input = {
    forward: keys['w'],
    backward: keys['s'],
    left: keys['a'],
    right: keys['d'],
    sprint: keys['shift'],
    angle: calculateAimAngle()
  };
  
  // 2. PREDICTION PHASE
  // Move player locally without waiting for server
  predictPlayerMovement(input);
  
  // 3. RENDER PHASE
  clearCanvas();
  drawGameWorld();
  drawPlayers();
  drawBullets();
  drawHUD();
  
  // 4. NETWORK PHASE (async, non-blocking)
  if (shouldSendUpdate()) {
    socket.emit('playerInput', input);
  }
  
  requestAnimationFrame(gameLoop);
}
```

---

## Data Structures

### Player Object

```javascript
class Player {
  // Identification
  id: string                 // Socket ID
  username: string          // Display name
  roomId: string            // Current room
  
  // Position & Movement
  x: number                 // X coordinate (game units)
  y: number                 // Y coordinate (game units)
  vx: number                // X velocity
  vy: number                // Y velocity
  angle: number             // Aiming angle (radians)
  
  // Combat
  health: number            // Current HP (0-100)
  maxHealth: number         // Max HP
  armor: number             // Armor rating (0-100)
  maxArmor: number          // Max armor
  
  // Weapons
  currentWeapon: number     // Weapon slot (0-3)
  weapons: [number, ...]    // Equipped weapon IDs
  ammo: [number, ...]       // Ammo per weapon
  grenades: number          // Grenade count
  
  // State
  isDead: boolean           // Dead flag
  isSprinting: boolean      // Sprint state
  isCrouching: boolean      // Crouch state
  isAiming: boolean         // ADS state
  
  // Stats
  kills: number             // Total kills
  deaths: number            // Total deaths
  damage: number            // Damage dealt
  accuracy: number          // Hit accuracy %
  
  // Timing
  lastShot: number          // Timestamp of last shot
  lastReload: number        // Timestamp of last reload
  spawnTime: number         // Spawn timestamp
}
```

### GameRoom Object

```javascript
class GameRoom {
  // Metadata
  id: string                        // Room UUID
  name: string                      // Display name
  maxPlayers: number                // Max capacity
  
  // Players
  players: Map<id, Player>          // Active players
  
  // Game State
  gameState: "waiting|playing|ended"
  mapId: number                     // Current map
  roundStartTime: number            // Round start timestamp
  roundDuration: number             // Duration in seconds
  
  // Scoring
  teamAScore: number                // Team A score
  teamBScore: number                // Team B score
  
  // Statistics
  totalKills: number                // Cumulative kills
  totalDeaths: number               // Cumulative deaths
}
```

### Weapon Object

```javascript
{
  id: number,                  // Weapon ID
  name: string,                // Display name
  type: "ar|smg|shotgun|sniper|pistol|grenade"
  
  // Ballistics
  damage: number,              // Base damage
  fireRate: number,            // Fire rate (ms)
  range: number,               // Effective range (units)
  magazine: number,            // Magazine size
  reloadTime: number,          // Reload time (seconds)
  
  // Accuracy
  spread: number,              // Bullet spread (radians)
  recoil: number,              // Recoil magnitude
  
  // Economy
  price: number,               // Buy price
  rarity: "common|uncommon|rare|epic|legendary"
}
```

---

## State Management

### Client-Side Prediction

Problem: Network latency causes lag

Solution: Predict locally, reconcile with server

```javascript
class ClientPrediction {
  constructor() {
    this.predictedState = null;
    this.confirmedState = null;
    this.inputBuffer = [];
  }
  
  update(input, deltaTime) {
    // 1. Apply input locally (prediction)
    this.predictedState.x += input.forward ? speed : 0;
    this.predictedState.y += input.backward ? speed : 0;
    
    // Buffer input for server
    this.inputBuffer.push({
      input,
      timestamp: Date.now()
    });
  }
  
  onServerState(serverState) {
    // 2. Receive confirmed state from server
    this.confirmedState = serverState;
    
    // 3. Check if prediction was correct
    const error = distance(
      this.predictedState,
      this.confirmedState
    );
    
    if (error > THRESHOLD) {
      // Large error: reconcile
      this.reconcile(serverState);
    }
  }
  
  reconcile(serverState) {
    // Snap to server + replay inputs
    this.predictedState = serverState;
    
    // Replay buffered inputs
    this.inputBuffer.forEach(item => {
      this.update(item.input, DELTA_TIME);
    });
    
    // Clear buffer
    this.inputBuffer = [];
  }
}
```

### Interpolation

Smoothly display remote players between updates:

```javascript
class Interpolation {
  update(remotePlayer, serverUpdate) {
    // Store previous position
    const oldX = remotePlayer.x;
    const oldY = remotePlayer.y;
    
    // New target
    const newX = serverUpdate.x;
    const newY = serverUpdate.y;
    
    // Calculate required speed
    const distance = Math.hypot(newX - oldX, newY - oldY);
    const updateInterval = TICK_INTERVAL; // 33.33ms at 30 TPS
    
    remotePlayer.targetX = newX;
    remotePlayer.targetY = newY;
    remotePlayer.interpSpeed = distance / updateInterval;
  }
  
  render(remotePlayer, deltaTime) {
    // Move toward target smoothly
    const dx = remotePlayer.targetX - remotePlayer.x;
    const dy = remotePlayer.targetY - remotePlayer.y;
    
    if (Math.hypot(dx, dy) > 1) {
      const dist = Math.hypot(dx, dy);
      remotePlayer.x += (dx / dist) * remotePlayer.interpSpeed * deltaTime;
      remotePlayer.y += (dy / dist) * remotePlayer.interpSpeed * deltaTime;
    }
  }
}
```

---

## Anti-Cheat

### Server-Authoritative Validation

```javascript
function onShoot(player, shootData) {
  // 1. Validate weapon
  if (!player.weapons.includes(shootData.weaponId)) {
    return REJECT("Invalid weapon");
  }
  
  // 2. Check ammo
  if (player.ammo[player.currentWeapon] <= 0) {
    return REJECT("Out of ammo");
  }
  
  // 3. Validate fire rate
  const fireRate = WEAPONS[shootData.weaponId].fireRate;
  const timeSinceLastShot = Date.now() - player.lastShot;
  
  if (timeSinceLastShot < fireRate) {
    return REJECT("Fire rate too fast (possible aimbot)");
  }
  
  // 4. Validate angle (anti-aimbot)
  const angleDelta = Math.abs(shootData.angle - player.lastAngle);
  const maxSnapAngle = 0.5; // radians, ~28 degrees
  
  if (angleDelta > maxSnapAngle) {
    return REJECT("Angle snap detected (aimbot?)");
  }
  
  // 5. Validate position (anti-speedhack)
  const maxDistance = PLAYER_MAX_SPEED * timeSinceLastShot;
  const distance = Math.hypot(
    player.x - player.lastX,
    player.y - player.lastY
  );
  
  if (distance > maxDistance) {
    return REJECT("Position moved too fast (speedhack)");
  }
  
  // 6. Hitscan detection
  const hits = calculateHits(
    shootData,
    getOtherPlayers(player.roomId)
  );
  
  // 7. Apply damage server-side
  hits.forEach(hit => {
    const target = getPlayer(hit.targetId);
    target.takeDamage(hit.damage);
  });
  
  // 8. Broadcast to all clients
  io.to(player.roomId).emit('bulletFired', {
    playerId: player.id,
    x: player.x,
    y: player.y,
    angle: shootData.angle,
    hits: hits
  });
  
  player.lastShot = Date.now();
  player.lastAngle = shootData.angle;
  player.lastX = player.x;
  player.lastY = player.y;
}
```

### Anti-Speedhack Detection

```javascript
function validateMovement(player, newX, newY, deltaTime) {
  const distance = Math.hypot(
    newX - player.x,
    newY - player.y
  );
  
  const maxDistance = PLAYER_MAX_SPEED * deltaTime;
  
  if (distance > maxDistance * 1.1) {  // 10% tolerance for lag
    logger.warn(`Speedhack detected: ${player.username}`);
    player.flags.suspiciousMovement++;
    
    // Auto-ban after threshold
    if (player.flags.suspiciousMovement > 10) {
      banPlayer(player.id);
    }
    
    // Revert position
    return player.lastValidX, player.lastValidY;
  }
  
  return newX, newY;
}
```

---

## Performance Optimization

### Object Pooling

```javascript
class ObjectPool {
  constructor(ObjectClass, initialSize = 100) {
    this.available = [];
    this.inUse = new Set();
    
    for (let i = 0; i < initialSize; i++) {
      this.available.push(new ObjectClass());
    }
  }
  
  acquire() {
    const obj = this.available.pop();
    if (!obj) {
      return new this.constructor();
    }
    this.inUse.add(obj);
    return obj;
  }
  
  release(obj) {
    obj.reset();
    this.inUse.delete(obj);
    this.available.push(obj);
  }
}

// Usage
const bulletPool = new ObjectPool(Bullet, 1000);

function fireWeapon(player) {
  const bullet = bulletPool.acquire();
  bullet.init(player.x, player.y, player.angle);
  // ...
  
  // Later:
  bulletPool.release(bullet);
}
```

### Network Compression

```javascript
function compressPlayerState(player) {
  // Instead of sending full object
  // {x: 45.231, y: 67.891, health: 85, kills: 3, ...}
  
  // Send compressed bytes
  // Using bit-packing
  return {
    id: player.id.slice(0, 8), // Short ID
    p: [Math.floor(player.x * 2), Math.floor(player.y * 2)], // Quantized
    h: player.health, // Uint8
    a: player.angle & 0xFF, // Quantized angle
    k: player.kills, // Uint8
    f: [player.isSprinting ? 1 : 0] // Flags
  };
}
```

### Spatial Hashing (Bullet Checks)

```javascript
class SpatialGrid {
  constructor(cellSize = 20) {
    this.cellSize = cellSize;
    this.grid = new Map();
  }
  
  getCell(x, y) {
    return `${Math.floor(x/this.cellSize)}_${Math.floor(y/this.cellSize)}`;
  }
  
  add(object) {
    const cell = this.getCell(object.x, object.y);
    if (!this.grid.has(cell)) {
      this.grid.set(cell, []);
    }
    this.grid.get(cell).push(object);
  }
  
  getNearby(x, y, range) {
    // Only check nearby cells
    const nearby = [];
    for (let dx = -1; dx <= 1; dx++) {
      for (let dy = -1; dy <= 1; dy++) {
        const cell = `${Math.floor(x/this.cellSize) + dx}_${Math.floor(y/this.cellSize) + dy}`;
        if (this.grid.has(cell)) {
          nearby.push(...this.grid.get(cell));
        }
      }
    }
    return nearby;
  }
}

// Usage: Faster bullet-to-player checks
const gridHits = spatialGrid.getNearby(bullet.x, bullet.y, 5);
```

---

## Scalability

### Horizontal Scaling Architecture

```
                    ┌─────────────────┐
                    │  Load Balancer  │
                    │   (nginx/HAProxy)│
                    └────────┬─────────┘
                             │
         ┌───────────────────┼───────────────────┐
         │                   │                   │
    ┌────▼─────┐        ┌────▼─────┐        ┌────▼─────┐
    │ Game S. 1│        │ Game S. 2│        │ Game S. 3│
    │(Node 1)  │        │(Node 2)  │        │(Node 3)  │
    │Port 3000 │        │Port 3001 │        │Port 3002 │
    └────┬─────┘        └────┬─────┘        └────┬─────┘
         │                   │                   │
         └───────────────────┼───────────────────┘
                             │
                    ┌────────▼────────┐
                    │   Redis Pub/Sub  │
                    │   (Room Sync)    │
                    └──────────────────┘
```

### Recommendation for Growth

- **<100 players**: Single server sufficient
- **100-500 players**: Use Redis for room sync
- **500-5000 players**: Multiple servers + load balancer
- **5000+ players**: Microservices architecture

---

## Security Checklist

- [ ] Enable HTTPS/TLS for production
- [ ] Validate all client inputs
- [ ] Rate limit Socket.io events
- [ ] Implement JWT for authentication
- [ ] Use environment variables for secrets
- [ ] Regular security audits
- [ ] Player ban system
- [ ] DDoS protection (Cloudflare)
- [ ] Encrypted password storage
- [ ] CORS configured properly

---

## Conclusion

This architecture provides:

✅ **Low Latency** - Client prediction + server authority
✅ **Security** - Server-authoritative validation
✅ **Scalability** - Modular design, ready for growth
✅ **Performance** - Object pooling, spatial hashing
✅ **Fairness** - Anti-cheat detection
✅ **Maintainability** - Clean separation of concerns

---

For questions, see [README.md](README.md)
