// MULTIPLAYER WEB SHOOTER - SERVER
// Node.js + WebSocket Server
// Run: node server.js

const express = require('express');
const http = require('http');
const WebSocket = require('ws');
const cors = require('cors');
const jwt = require('jsonwebtoken');
const path = require('path');

const app = express();
const server = http.createServer(app);
const wss = new WebSocket.Server({ server });

// Middleware
app.use(cors());
app.use(express.json());
app.use(express.static(path.join(__dirname, 'public')));

const SECRET_KEY = 'ultra-secret-game-key-12345';
const TICK_RATE = 60; // Server ticks per second
const PLAYER_UPDATE_RATE = 30; // Player update rate

// Game state
const rooms = new Map();
const players = new Map();
const activePlayers = new Map();

// Game configuration
const GAME_CONFIG = {
  maxPlayersPerRoom: 8,
  roomIdLength: 6,
  playerSpeed: 0.15,
  sprintSpeed: 0.25,
  crouchSpeed: 0.08,
  jumpPower: 0.5,
  gravity: 0.01,
  maxHealth: 100,
  respawnTime: 3000,
  tickRate: TICK_RATE
};

// Room management
class GameRoom {
  constructor(roomId, isPrivate = false) {
    this.roomId = roomId;
    this.isPrivate = isPrivate;
    this.players = new Map();
    this.gameState = {
      mapId: 'arena_01',
      matchTime: 0,
      maxTime: 600,
      inProgress: false
    };
    this.matchData = {
      kills: new Map(),
      deaths: new Map(),
      startTime: Date.now()
    };
  }

  addPlayer(playerId, player) {
    if (this.players.size >= GAME_CONFIG.maxPlayersPerRoom) {
      return false;
    }
    this.players.set(playerId, player);
    this.matchData.kills.set(playerId, 0);
    this.matchData.deaths.set(playerId, 0);
    return true;
  }

  removePlayer(playerId) {
    this.players.delete(playerId);
  }

  getPlayers() {
    return Array.from(this.players.values());
  }

  isEmpty() {
    return this.players.size === 0;
  }

  isFull() {
    return this.players.size >= GAME_CONFIG.maxPlayersPerRoom;
  }
}

// Player class
class Player {
  constructor(playerId, username, ws) {
    this.id = playerId;
    this.username = username;
    this.ws = ws;
    this.roomId = null;

    // Position & Movement
    this.position = { x: 0, y: 0, z: 0 };
    this.velocity = { x: 0, y: 0, z: 0 };
    this.rotation = { yaw: 0, pitch: 0 };
    this.state = {
      moving: false,
      sprinting: false,
      crouching: false,
      aiming: false,
      jumping: false,
      sliding: false
    };

    // Combat
    this.health = GAME_CONFIG.maxHealth;
    this.armor = 0;
    this.currentWeapon = 'ar_15';
    this.weapons = {
      ar_15: { ammo: 30, magazine: 30, reserve: 120 },
      smg_mp5: { ammo: 25, magazine: 25, reserve: 75 },
      shotgun: { ammo: 8, magazine: 8, reserve: 32 },
      sniper: { ammo: 5, magazine: 5, reserve: 20 },
      pistol: { ammo: 12, magazine: 12, reserve: 48 }
    };
    this.reloadingTime = 0;
    this.lastShotTime = 0;
    this.isReloading = false;
    this.isDead = false;
    this.deathTime = 0;

    // Anti-cheat
    this.lastValidatedPosition = { x: 0, y: 0, z: 0, time: Date.now() };
    this.suspiciousActions = 0;
    this.tickCounter = 0;
  }

  update(input) {
    if (this.isDead && Date.now() - this.deathTime > GAME_CONFIG.respawnTime) {
      this.respawn();
    }

    if (this.isDead) return;

    // Validate input
    if (!this.validateInput(input)) {
      this.suspiciousActions++;
      if (this.suspiciousActions > 5) {
        console.log(`[ANTI-CHEAT] Player ${this.id} flagged for suspicious activity`);
      }
    }

    // Update state from input
    this.state.moving = input.moving || false;
    this.state.sprinting = input.sprinting || false;
    this.state.crouching = input.crouching || false;
    this.state.aiming = input.aiming || false;
    this.state.jumping = input.jumping || false;

    this.rotation.yaw = input.rotation?.yaw || 0;
    this.rotation.pitch = input.rotation?.pitch || 0;

    // Movement
    const moveInput = input.moveInput || { x: 0, z: 0 };
    let speed = GAME_CONFIG.playerSpeed;
    if (this.state.sprinting) speed = GAME_CONFIG.sprintSpeed;
    if (this.state.crouching) speed = GAME_CONFIG.crouchSpeed;

    // Apply velocity
    this.velocity.x += moveInput.x * speed;
    this.velocity.z += moveInput.z * speed;

    // Friction
    this.velocity.x *= 0.92;
    this.velocity.z *= 0.92;

    // Gravity
    this.velocity.y -= GAME_CONFIG.gravity;

    // Apply velocity to position
    this.position.x += this.velocity.x;
    this.position.y += this.velocity.y;
    this.position.z += this.velocity.z;

    // Boundary check
    this.position.x = Math.max(-50, Math.min(50, this.position.x));
    this.position.z = Math.max(-50, Math.min(50, this.position.z));

    // Ground collision
    if (this.position.y < 0) {
      this.position.y = 0;
      this.velocity.y = 0;
    }

    // Weapon handling
    if (input.shooting && !this.isReloading) {
      this.shoot();
    }

    if (input.reloading) {
      this.reload();
    }

    if (input.switchWeapon) {
      this.currentWeapon = input.switchWeapon;
    }

    // Update reload timer
    if (this.isReloading) {
      this.reloadingTime--;
      if (this.reloadingTime <= 0) {
        this.isReloading = false;
        const weapon = this.weapons[this.currentWeapon];
        const ammoNeeded = weapon.magazine - weapon.ammo;
        const ammoAvailable = Math.min(ammoNeeded, weapon.reserve);
        weapon.ammo += ammoAvailable;
        weapon.reserve -= ammoAvailable;
      }
    }
  }

  validateInput(input) {
    // Anti-speedhack: Check if movement is reasonable
    const timeDelta = (Date.now() - this.lastValidatedPosition.time) / 1000;
    const maxDistance = GAME_CONFIG.sprintSpeed * timeDelta * 1.5;
    
    const distance = Math.hypot(
      this.position.x - this.lastValidatedPosition.x,
      this.position.z - this.lastValidatedPosition.z
    );

    if (distance > maxDistance && timeDelta > 0) {
      return false;
    }

    this.lastValidatedPosition = {
      x: this.position.x,
      z: this.position.z,
      time: Date.now()
    };

    return true;
  }

  shoot() {
    const weapon = this.weapons[this.currentWeapon];
    if (weapon.ammo <= 0) return;
    if (Date.now() - this.lastShotTime < 50) return; // Fire rate check

    weapon.ammo--;
    this.lastShotTime = Date.now();
  }

  reload() {
    if (this.isReloading) return;
    const weapon = this.weapons[this.currentWeapon];
    if (weapon.ammo === weapon.magazine) return;
    
    this.isReloading = true;
    this.reloadingTime = 60; // 1 second at 60 TPS
  }

  takeDamage(damage) {
    const armorReduction = Math.min(this.armor, damage * 0.25);
    const finalDamage = damage - armorReduction;
    this.armor = Math.max(0, this.armor - armorReduction);
    this.health -= finalDamage;

    if (this.health <= 0) {
      this.health = 0;
      this.isDead = true;
      this.deathTime = Date.now();
    }
  }

  respawn() {
    this.health = GAME_CONFIG.maxHealth;
    this.armor = 0;
    this.isDead = false;
    this.position = { x: 0, y: 5, z: 0 };
    this.velocity = { x: 0, y: 0, z: 0 };
  }

  getState() {
    return {
      id: this.id,
      username: this.username,
      position: this.position,
      rotation: this.rotation,
      state: this.state,
      health: this.health,
      armor: this.armor,
      currentWeapon: this.currentWeapon,
      ammo: this.weapons[this.currentWeapon].ammo,
      isReloading: this.isReloading,
      isDead: this.isDead
    };
  }
}

// Utility functions
function generateRoomId() {
  return Math.random().toString(36).substring(2, 8).toUpperCase();
}

function generatePlayerId() {
  return 'player_' + Date.now() + '_' + Math.random().toString(36).substr(2, 9);
}

function generateToken(playerId, username) {
  return jwt.sign({ playerId, username }, SECRET_KEY, { expiresIn: '24h' });
}

// API Routes
app.post('/api/auth/login', (req, res) => {
  const { username } = req.body;
  if (!username || username.length < 2) {
    return res.status(400).json({ error: 'Invalid username' });
  }

  const playerId = generatePlayerId();
  const token = generateToken(playerId, username);

  res.json({
    playerId,
    username,
    token
  });
});

app.post('/api/rooms/create', (req, res) => {
  const { isPrivate } = req.body;
  const roomId = generateRoomId();
  
  const room = new GameRoom(roomId, isPrivate || false);
  rooms.set(roomId, room);

  res.json({
    roomId,
    isPrivate: room.isPrivate,
    playersCount: 0,
    maxPlayers: GAME_CONFIG.maxPlayersPerRoom
  });
});

app.get('/api/rooms/list', (req, res) => {
  const roomsList = Array.from(rooms.values())
    .filter(room => !room.isPrivate && room.players.size < GAME_CONFIG.maxPlayersPerRoom)
    .map(room => ({
      roomId: room.roomId,
      playersCount: room.players.size,
      maxPlayers: GAME_CONFIG.maxPlayersPerRoom,
      mapId: room.gameState.mapId,
      inProgress: room.gameState.inProgress
    }));

  res.json(roomsList);
});

app.get('/api/config/weapons', (req, res) => {
  res.json({
    ar_15: {
      name: 'AR-15',
      type: 'assault_rifle',
      damage: 28,
      fireRate: 750,
      range: 800,
      accuracy: 0.8,
      recoil: 2.5,
      magazine: 30,
      reloadTime: 2.3,
      spread: 0.8
    },
    smg_mp5: {
      name: 'MP5',
      type: 'smg',
      damage: 18,
      fireRate: 950,
      range: 300,
      accuracy: 0.7,
      recoil: 1.8,
      magazine: 25,
      reloadTime: 1.8,
      spread: 1.5
    },
    shotgun: {
      name: 'Shotgun',
      type: 'shotgun',
      damage: 75,
      fireRate: 80,
      range: 200,
      accuracy: 0.5,
      recoil: 5,
      magazine: 8,
      reloadTime: 3.0,
      spread: 8,
      pellets: 8
    },
    sniper: {
      name: 'Sniper Rifle',
      type: 'sniper',
      damage: 90,
      fireRate: 40,
      range: 1500,
      accuracy: 0.95,
      recoil: 6,
      magazine: 5,
      reloadTime: 3.5,
      spread: 0.2,
      zoomLevel: 8
    },
    pistol: {
      name: 'Pistol',
      type: 'pistol',
      damage: 22,
      fireRate: 400,
      range: 400,
      accuracy: 0.75,
      recoil: 1.5,
      magazine: 12,
      reloadTime: 1.2,
      spread: 1.2
    }
  });
});

// WebSocket handling
wss.on('connection', (ws) => {
  let player = null;
  let currentRoom = null;

  console.log('New WebSocket connection');

  ws.on('message', (data) => {
    try {
      const message = JSON.parse(data);

      switch (message.type) {
        case 'auth':
          handleAuth(message, ws);
          break;

        case 'joinRoom':
          handleJoinRoom(message, ws);
          break;

        case 'playerInput':
          if (player && currentRoom) {
            player.update(message.input);
          }
          break;

        case 'shot':
          handleShot(message, currentRoom, player);
          break;

        case 'leaveRoom':
          handleLeaveRoom(currentRoom, player);
          break;
      }
    } catch (error) {
      console.error('Message error:', error);
    }
  });

  ws.on('close', () => {
    if (player && currentRoom) {
      handleLeaveRoom(currentRoom, player);
    }
  });

  function handleAuth(message, ws) {
    const token = message.token;
    try {
      const decoded = jwt.verify(token, SECRET_KEY);
      player = new Player(decoded.playerId, decoded.username, ws);
      players.set(player.id, player);

      ws.send(JSON.stringify({
        type: 'authSuccess',
        playerId: player.id,
        username: player.username
      }));
    } catch (error) {
      ws.send(JSON.stringify({ type: 'authError', error: 'Invalid token' }));
      ws.close();
    }
  }

  function handleJoinRoom(message, ws) {
    let room = rooms.get(message.roomId);

    if (!room) {
      ws.send(JSON.stringify({ type: 'error', error: 'Room not found' }));
      return;
    }

    if (room.isFull()) {
      ws.send(JSON.stringify({ type: 'error', error: 'Room is full' }));
      return;
    }

    // Spawn player at spawn point
    player.position = {
      x: (Math.random() - 0.5) * 40,
      y: 5,
      z: (Math.random() - 0.5) * 40
    };

    if (room.addPlayer(player.id, player)) {
      player.roomId = message.roomId;
      currentRoom = room;

      ws.send(JSON.stringify({
        type: 'joinSuccess',
        roomId: message.roomId,
        playerId: player.id
      }));

      // Broadcast player joined
      broadcastToRoom(currentRoom, {
        type: 'playerJoined',
        player: player.getState()
      });
    }
  }

  function handleLeaveRoom(room, player) {
    if (room && player) {
      room.removePlayer(player.id);
      
      broadcastToRoom(room, {
        type: 'playerLeft',
        playerId: player.id
      });

      if (room.isEmpty()) {
        rooms.delete(room.roomId);
      }
    }
  }

  function handleShot(message, room, player) {
    if (!room || !player) return;

    // Server-side hit detection (anti-cheat)
    const targetPlayer = room.players.get(message.targetId);
    if (!targetPlayer || targetPlayer.isDead) return;

    // Validate shot distance and angle
    const distance = Math.hypot(
      targetPlayer.position.x - player.position.x,
      targetPlayer.position.z - player.position.z
    );

    const weapon = require('./config/weapons.json')[player.currentWeapon];
    if (distance > weapon.range) return;

    let damage = weapon.damage;
    if (message.isHeadshot) {
      damage *= 2; // Headshot multiplier
    }

    targetPlayer.takeDamage(damage);
    room.matchData.kills.set(player.id, (room.matchData.kills.get(player.id) || 0) + 1);
    room.matchData.deaths.set(targetPlayer.id, (room.matchData.deaths.get(targetPlayer.id) || 0) + 1);

    broadcastToRoom(room, {
      type: 'hit',
      shooterId: player.id,
      targetId: targetPlayer.id,
      damage: damage,
      isHeadshot: message.isHeadshot,
      targetHealth: targetPlayer.health
    });
  }
});

// Game loop
setInterval(() => {
  for (const room of rooms.values()) {
    // Update game state
    const playersState = room.getPlayers().map(p => p.getState());

    // Send updates to all players in room
    broadcastToRoom(room, {
      type: 'gameState',
      players: playersState,
      matchData: {
        kills: Object.fromEntries(room.matchData.kills),
        deaths: Object.fromEntries(room.matchData.deaths)
      }
    });
  }
}, 1000 / PLAYER_UPDATE_RATE);

function broadcastToRoom(room, message) {
  const data = JSON.stringify(message);
  for (const player of room.players.values()) {
    if (player.ws.readyState === WebSocket.OPEN) {
      player.ws.send(data);
    }
  }
}

// Start server
const PORT = process.env.PORT || 8080;
server.listen(PORT, () => {
  console.log(`
╔════════════════════════════════════════════════════════╗
║   MULTIPLAYER WEB SHOOTER - SERVER                    ║
║   Server running on: ws://localhost:${PORT}              ║
║   API running on: http://localhost:${PORT}              ║
╚════════════════════════════════════════════════════════╝
  `);
});

module.exports = { server, rooms, players };
